UNIT Win95LFN;

(*    Long filename support for DOS 16-bit Pascal programs V1.0
      =========================================================

      by I. Eckel / Bad Soden / Germany

      This unit provides basic Windows95 long filename support
      for programs written in Borland Pascal 7.0. It implements
      basic functions to find, rename, convert, access attributes
      and delete files or directories with long filenames under
      Windows95.


      The following functions are implemented:

      FUNCTION  GetWinVersion : WORD;
      Access the version of the active MS-Windows system (4 = Windows95).
      The version number is stored in the variable WinVersion.

      FUNCTION  LFN_FindFirst(Path : STRING; SAttr: BYTE; VAR Srec : LFN_SearchRec) : BOOLEAN;
      Search the first file or directory matching the search-specification. Returns
      the record LFN_Searchrec containing the short filename (8.3), long filename
      and the extended time attributes.

      FUNCTION  LFN_FindNext(VAR Srec : LFN_SearchRec) : BOOLEAN;
      Search the next file or directory matching the search-specification
      of the last LFN_FindFirst call.

      PROCEDURE LFN_FindClose(Srec : LFN_SearchRec);
      Close the searchhandle created by LFN_Findfirst.
      LFN_FindClose should only be called if the last call of LFN_FindFirst
      or LFN_Findnext returned a DosError = 0. This procedure should
      not be called, if the last call of LFN_FindFirst or LFN_FindNext
      has returned a DosError > 0. Otherwise you get the DosError
      "Invalid handle".

      FUNCTION  LFN_AccessAttributes(Path : STRING; WichTime : BYTE; VAR Time: LONGINT) : BOOLEAN;
      Access the extended time-attributes of Windows95 creationtime,
      time of last read access and last time of write access.
      Valid parameters for Wichtime are:
      SetModificationTime = 3;
      GetModificationTime = 4;
      SetAccessTime       = 5;
      GetAccessTime       = 6;
      SetCreationTime     = 7;
      GetCreationTime     = 8;

      FUNCTION  LFN_GetDir(DriveNo : BYTE; VAR Dir : STRING) : BOOLEAN;
      Read the working directory with long filename.

      FUNCTION  LFN_CreateDir(Name : STRING) : BOOLEAN;
      Create a new directory with long filename.

      FUNCTION  LFN_RemoveDir(Name : STRING) : BOOLEAN;
      Delete a directory with long filename.

      FUNCTION  LFN_RenameFile(OldName, NewName : STRING) : BOOLEAN;
      Rename a file containing a short or long filename to a new short or
      long filename. A file could also be moved within a logical drive.

      FUNCTION  LFN_CreateFile(Name : STRING; Attr : BYTE) : BOOLEAN;
      Creates a new file with long filename and close it.

      FUNCTION  LFN_DeleteFile(Name : STRING) : BOOLEAN;
      Delete a file with long filename.

      FUNCTION  LFN_FileExist(Path : STRING; SAttr : BYTE) : BOOLEAN;
      Returns True, if long filename exists.

      FUNCTION  LFN_GetLongFileName(SName : PATHSTR; VAR LongName : STRING) : BOOLEAN;
      Read the long filename of a given short filename (like TRUENAME,
      but strips the leading path).

      FUNCTION  LFN_GetShortFileName(LongName : STRING; VAR ShortName : String12) : BOOLEAN;
      Read the short filename of a given long filename (like TRUENAME,
      but strips the leading path).

      PROCEDURE LFN_FileSplit(Path : STRING; VAR Dir, Name : STRING; VAR Ext : EXTSTR);
      Splits a complete path into directory, name and extension. Works
      like the Borland Pascal procedure FSPLIT.

      All functions (except LFN_FileSplit) returns TRUE, if the action was
      successful, otherwise false. The DOS returncodes are stored in the
      variable DOSERROR of the unit DOS. The variable LFNSupport contains
      TRUE, if the running operation system supports long filenames (not OS/2!)

      --------------------------------------------------------------------

      Copyright:
      This unit is freeware and may be distributed freely without any fee.
      Please report bugs to the following postal or email address:

      Ingo Eckel
      Sodener Weg 38
      D-65812 Bad Soden
      Germany

      email:
      CompuServe: 100773,3172
      Internet: 100773.3172@compuserve.com                        *)


{$A+,B-,E-,F-,I-,N-,O-,V-}

{$IFDEF DEBUG}
{$L+,D+,S+,Q+,R+}
{$ELSE}
{$L-,D-,S-,Q-,R-}
{$ENDIF}


INTERFACE


USES DOS,
     STRINGS;

TYPE String12 = STRING[12];

TYPE LFN_SearchRec = RECORD
     Attr      : BYTE;     { Attributes            }
     CreTime   : LONGINT;  { Creationtime          }
     ModTime   : LONGINT;  { Last write access time}
     AccTime   : LONGINT;  { Last read access time }
     Size      : LONGINT;  { Filesize in bytes     }
     ShortName : String12; { DOS 8.3 filename      }
     LongName  : STRING;   { Win95 long filename   }
     HANDLE    : WORD;     { Handle for FindNext/FindClose}
   END;

CONST SetModificationTime = 3;
      GetModificationTime = 4;
      SetAccessTime       = 5;
      GetAccessTime       = 6;
      SetCreationTime     = 7;
      GetCreationTime     = 8;

      LFNSupport : BOOLEAN = FALSE;

VAR   WinVersion : WORD;


FUNCTION  GetWinVersion : WORD;
FUNCTION  LFN_FindFirst(Path : STRING; SAttr: BYTE; VAR Srec : LFN_SearchRec) : BOOLEAN;
FUNCTION  LFN_FindNext(VAR Srec : LFN_SearchRec) : BOOLEAN;
PROCEDURE LFN_FindClose(Srec : LFN_SearchRec);
FUNCTION  LFN_AccessAttributes(Path : STRING; WichTime : BYTE; VAR Time: LONGINT) : BOOLEAN;
FUNCTION  LFN_GetDir(DriveNo : BYTE; VAR Dir : STRING) : BOOLEAN;
FUNCTION  LFN_CreateDir(Name : STRING) : BOOLEAN;
FUNCTION  LFN_RemoveDir(Name : STRING) : BOOLEAN;
FUNCTION  LFN_RenameFile(OldName, NewName : STRING) : BOOLEAN;
FUNCTION  LFN_CreateFile(Name : STRING; Attr : BYTE) : BOOLEAN;
FUNCTION  LFN_DeleteFile(Name : STRING) : BOOLEAN;
FUNCTION  LFN_FileExist(Path : STRING; SAttr : BYTE) : BOOLEAN;
FUNCTION  LFN_GetLongFileName(SName : PATHSTR; VAR LongName : STRING) : BOOLEAN;
FUNCTION  LFN_GetShortFileName(LongName : STRING; VAR ShortName : String12) : BOOLEAN;
PROCEDURE LFN_FileSplit(Path : STRING; VAR Dir, Name : STRING; VAR Ext : EXTSTR);


IMPLEMENTATION


TYPE LFN_Srec_Type = RECORD
     Attr      : LONGINT;
     CreTime   : LONGINT;
     Unused1   : LONGINT;
     AccTime   : LONGINT;
     Unused2   : LONGINT;
     ModTime   : LONGINT;
     Unused3   : LONGINT;
     SizeHi    : LONGINT;
     SizeLo    : LONGINT;
     Reserved  : COMP;
     LongName  : ARRAY[1..260] OF CHAR;
     ShortName : ARRAY[1..14] OF CHAR;
     HANDLE    : WORD;
   END;
   
   
   
FUNCTION GetWinVersion : WORD;
VAR
    Regs : REGISTERS;    { to hold register info }
    
    { Routine to determine if Windows is currently running }
BEGIN  (* Win3X *)
  Regs.AX := $1600;
  INTR($2F, Regs);     { Call Int 2F }
  IF Regs.AL IN [$00,$80,$01,$FF] THEN { Check returned value }
  GetWinVersion := 0
  ELSE
  GetWinVersion := Regs.AX;
END; {GetWinVersion}


FUNCTION Minimum(A, B : Integer) : Integer;
BEGIN
  IF A > B THEN
  Minimum := B
  ELSE
  Minimum := A;
END; { Minimum }

FUNCTION Maximum(A ,B : Integer) : Integer;
BEGIN
  IF A < B THEN
  Maximum := B
  ELSE
  Maximum := A;
END; { Maximum }


FUNCTION LFN_FindFirst(Path : STRING; SAttr: BYTE; VAR Srec : LFN_SearchRec) : BOOLEAN;

VAR Regs     : REGISTERS;
    LFN_SRec : LFN_Srec_Type;
    
    
BEGIN
  Path := Path + #0;
  WITH Regs DO
  BEGIN
    (* FindFirst *)
    AX := $714E;
    CL := SAttr;   (* Allowed attributes         *)
    CH := 0;       (* Required attributes        *)
    SI := 1;       (* Time format 1=DOS 0=Win95  *)
    DS := SEG(Path[1]);
    DX := OFS(Path[1]);
    ES := SEG(LFN_SRec);
    DI := OFS(LFN_SRec);
    MSDOS(Regs);
    
    FILLCHAR(Srec, SIZEOF(Srec),#0);
    
    IF (Regs.FLAGS AND FCARRY = 0) THEN
    BEGIN
      LFN_FindFirst := TRUE;
      DOSERROR := 0;
      Srec.Size := LFN_SRec.SizeLo;
      Srec.Attr := LFN_SRec.Attr AND $FF;
      Srec.ShortName := StrPas(ADDR(LFN_SRec.ShortName));
      Srec.LongName  := StrPas(ADDR(LFN_SRec.LongName));
      IF (Srec.ShortName = '') AND (LENGTH(Srec.LongName) <= 12) THEN
      Srec.ShortName := Srec.LongName;
      Srec.HANDLE    := Regs.AX;
      
      Srec.CreTime := LFN_SRec.CreTime;
      Srec.AccTime := LFN_SRec.AccTime;
      Srec.ModTime := LFN_SRec.ModTime;
    END
    ELSE
    BEGIN
      LFN_FindFirst := FALSE;
      DOSERROR := AX;
    END;
  END;
END; { LFN_FindFirst }



FUNCTION LFN_FindNext(VAR Srec : LFN_SearchRec) : BOOLEAN;

VAR Regs     : REGISTERS;
    LFN_SRec : LFN_Srec_Type;
    
BEGIN
  WITH Regs DO
  BEGIN
    (* FindNext *)
    AX := $714F;
    BX := Srec.HANDLE;   (* Allowed attributes         *)
    SI := 1;             (* Time format 1=DOS 0=Win95  *)
    ES := SEG(LFN_SRec);
    DI := OFS(LFN_SRec);
    MSDOS(Regs);
    
    IF (Regs.FLAGS AND FCARRY = 0) THEN
    BEGIN
      LFN_FindNext := TRUE;
      DOSERROR := 0;
      Srec.Size := LFN_SRec.SizeLo;
      Srec.Attr := LFN_SRec.Attr AND $FF;
      Srec.ShortName := StrPas(ADDR(LFN_SRec.ShortName));
      Srec.LongName  := StrPas(ADDR(LFN_SRec.LongName));
      IF (Srec.ShortName = '') AND (LENGTH(Srec.LongName) <= 12) THEN
      Srec.ShortName := Srec.LongName;
      
      Srec.CreTime := LFN_SRec.CreTime;
      Srec.AccTime := LFN_SRec.AccTime;
      Srec.ModTime := LFN_SRec.ModTime;
    END
    ELSE
    BEGIN
      LFN_FindNext := FALSE;
      DOSERROR := Regs.AX;
    END;
  END;
END; { LFN_FindNext }

PROCEDURE LFN_FindClose(Srec : LFN_SearchRec);

VAR Regs : REGISTERS;
BEGIN
  (* FindClose *)
  Regs.BX := Srec.HANDLE;
  Regs.AX := $71A1;
  MSDOS(Regs);
END; {LFN_FindClose}



FUNCTION  LFN_AccessAttributes(Path : STRING; WichTime : BYTE; VAR Time: LONGINT) : BOOLEAN;

VAR Regs     : REGISTERS;
    
BEGIN
  Path := Path + #0;
  WITH Regs DO
  BEGIN
    (* Access Extended Attributes *)
    AX := $7143;
    BL := WichTime;      (* Action *)
    DS := SEG(Path[1]);
    DX := OFS(Path[1]);
    
    CASE WichTime OF
      SetModificationTime,
      SetCreationTime: BEGIN
        Regs.DI := Time SHR 16;
        Regs.CX := Time AND $FFFF;
        Regs.SI := 0;
      END;
      SetAccessTime  : Regs.DI := Time SHR 16;
    END; {Case}
    MSDOS(Regs);
    
    
    IF (Regs.FLAGS AND FCARRY = 0) THEN
    BEGIN
      LFN_AccessAttributes := TRUE;
      DOSERROR := 0;
      CASE WichTime OF
        GetModificationTime,
        GetCreationTime : BEGIN
          Time := Regs.DI;
          Time := Time SHL 16 + Regs.CX;
        END;
        GetAccessTime   : BEGIN
          Time := Regs.DI;
          Time := Time SHL 16;
        END;
      END; {Case}
    END
    ELSE
    BEGIN
      LFN_AccessAttributes := FALSE;
      DOSERROR := AX;
    END;
  END;
END; { LFN_AccessAttributes }



FUNCTION LFN_RenameFile(OldName, NewName : STRING) : BOOLEAN;
VAR Regs   : REGISTERS;
    Result : INTEGER;
    
BEGIN
  LFN_RenameFile := FALSE;
  IF NOT LFNSupport THEN EXIT;
  OldName := OldName + #0;
  NewName := NewName + #0;
  Regs.AX := $7156;
  
  Regs.DS := SEG(OldName[1]);
  Regs.DX := OFS(OldName[1]);
  Regs.ES := SEG(NewName[1]);
  Regs.DI := OFS(NewName[1]);
  MSDOS(Regs);
  
  DOSERROR := Regs.AX;
  { Result = 5: Access denied = File allready exist }
  IF DOSERROR = $7100 THEN
  EXIT;
  IF Regs.FLAGS AND FCARRY = 0 THEN
  DOSERROR := 0
  ELSE {Fcarry <> 0}
  DOSERROR := Regs.AX;
  LFN_RenameFile := DOSERROR = 0;
END; { LFN_RenameFile }


FUNCTION LFN_GetDir(DriveNo : BYTE; VAR Dir : STRING) : BOOLEAN;
VAR Regs          : REGISTERS;
    ShortName     : STRING;
    LongNameArray : ARRAY[1..261] OF CHAR;
    ShortDir      : PATHSTR;
    
BEGIN
  LFN_GetDir := FALSE;
  IF NOT LFNSupport THEN EXIT;
  GETDIR(DriveNo, ShortDir);
  
  ShortDir:= ShortDir + #0;
  Regs.AX := $7160;
  Regs.CL := $2;
  Regs.CH := $0;
  Regs.DS := SEG(ShortDir[1]);
  Regs.SI := OFS(ShortDir[1]);
  Regs.ES := SEG(LongNameArray[1]);
  Regs.DI := OFS(LongNameArray[1]);
  MSDOS(Regs);
  IF Regs.FLAGS AND FCARRY <> 0 THEN
  BEGIN
    DOSERROR := Regs.AX;
    EXIT;
  END
  ELSE
  BEGIN
    DOSERROR := 0;
    Dir := StrPas(ADDR(LongNameArray));
  END;
  
  LFN_GetDir := DOSERROR = 0;
END; { LFN_GetDir }



FUNCTION LFN_CreateDir(Name : STRING) : BOOLEAN;

VAR Regs : REGISTERS;
    
BEGIN
  LFN_CreateDir := FALSE;
  IF NOT LFNSupport THEN EXIT;
  Name := Name + #0;
  Regs.AX := $7139;
  Regs.DS := SEG(Name[1]);
  Regs.DX := OFS(Name[1]);
  MSDOS(Regs);
  IF (Regs.FLAGS AND FCARRY = 0) THEN
  BEGIN
    DOSERROR := 0;
    LFN_CreateDir := TRUE;
  END
  ELSE
  BEGIN
    DOSERROR := Regs.AX;
    LFN_CreateDir := FALSE;
  END;
  
END; {LFN_CreateDir}



FUNCTION LFN_RemoveDir(Name : STRING) : BOOLEAN;

VAR Regs : REGISTERS;
    
BEGIN
  LFN_RemoveDir := FALSE;
  IF NOT LFNSupport THEN EXIT;
  Name := Name + #0;
  Regs.AX := $713A;
  Regs.DS := SEG(Name[1]);
  Regs.DX := OFS(Name[1]);
  MSDOS(Regs);
  IF (Regs.FLAGS AND FCARRY = 0) THEN
  BEGIN
    DOSERROR := 0;
    LFN_RemoveDir := TRUE;
  END
  ELSE
  BEGIN
    DOSERROR := Regs.AX;
    LFN_RemoveDir := FALSE;
  END;
END; {LFN_RemoveDir}



FUNCTION LFN_CreateFile(Name : STRING; Attr : BYTE) : BOOLEAN;

VAR Regs : REGISTERS;
    
BEGIN
  LFN_CreateFile := FALSE;
  IF NOT LFNSupport THEN EXIT;
  Name := Name + #0;
  Regs.AX := $716C;    (* Create file with long filename *)
  Regs.BX := 2; {Write-Only}
  Regs.CX := Attr;
  Regs.DX := 16;  {Create-Only}
  Regs.DS := SEG(Name[1]);
  Regs.SI := OFS(Name[1]);
  Regs.DI := 0;
  MSDOS(Regs);
  IF (Regs.FLAGS AND FCARRY = 0) THEN
  BEGIN
    DOSERROR := 0;
    LFN_CreateFile := TRUE;
    Regs.BX := Regs.AX;          (* DOS handle *)
    Regs.AH := $3E;              (* Close file *)
    MSDOS(Regs);
    IF (Regs.FLAGS AND FCARRY <> 0) THEN
    BEGIN
      DOSERROR := Regs.AX;
      EXIT;
    END;
  END
  ELSE
  BEGIN
    DOSERROR := Regs.AX;
    LFN_CreateFile := FALSE;
  END;
  
END; {LFN_CreateFile}


FUNCTION LFN_DeleteFile(Name : STRING) : BOOLEAN;

VAR Regs : REGISTERS;
    
BEGIN
  LFN_DeleteFile := FALSE;
  IF NOT LFNSupport THEN EXIT;
  Name := Name + #0;
  Regs.AX := $7141;
  Regs.CX := 0;
  Regs.DX := 16;  {Create-Only}
  Regs.DS := SEG(Name[1]);
  Regs.DX := OFS(Name[1]);
  Regs.SI := 0;
  MSDOS(Regs);
  IF (Regs.FLAGS AND FCARRY = 0) THEN
  BEGIN
    DOSERROR := 0;
    LFN_DeleteFile := TRUE;
  END
  ELSE
  BEGIN
    DOSERROR := Regs.AX;
    LFN_DeleteFile := FALSE;
  END;
END; {LFN_DeleteFile}



FUNCTION LFN_FileExist(Path : STRING; SAttr : BYTE) : BOOLEAN;
TYPE Bytes8   = ARRAY[1..8] OF BYTE;
     Char260  = ARRAY[1..260] OF CHAR;
     Char14   = ARRAY[1..14] OF CHAR;
     
VAR Regs     : REGISTERS;
    LFN_SRec : LFN_Srec_Type;
    
BEGIN
  Path := Path + #0;
  WITH Regs DO
  BEGIN
    (* FindFirst *)
    AX := $714E;
    CL := SAttr;   (* Allowed attributes  *)
    CH := 0;                                      (* Required attributes *)
    SI := 1;                                      (* Time format         *)
    DS := SEG(Path[1]);
    DX := OFS(Path[1]);
    ES := SEG(LFN_SRec);
    DI := OFS(LFN_SRec);
    MSDOS(Regs);
    
    DOSERROR := AX;
    IF (Regs.FLAGS AND FCARRY = 0) THEN
    BEGIN
      LFN_FileExist := TRUE;
      DOSERROR := 0;
    END
    ELSE
    LFN_FileExist := FALSE;
    
    (* FindClose *)
    IF AX = 0 THEN
    BEGIN
      BX := AX;
      AX := $71A1;
      MSDOS(Regs);
    END;
  END;
END; { LFN_FileExist }





FUNCTION LFN_GetLongFileName(SName : PATHSTR; VAR LongName : STRING) : BOOLEAN;

VAR Regs : REGISTERS;
    ShortName : STRING;
    LongNameArray : ARRAY[1..261] OF CHAR;
    Dir       : DIRSTR;
    Name      : NAMESTR;
    Ext       : EXTSTR;
    LongDir   : STRING;
    LongPath  : STRING;

BEGIN

  LFN_GetLongFileName := FALSE;
  LongName := '';
  IF NOT LFNSupport THEN EXIT;

  ShortName := SName + #0;
  Regs.AX := $7160;
  Regs.CL := $2;
  Regs.CH := $0;
  Regs.DS := SEG(ShortName[1]);
  Regs.SI := OFS(ShortName[1]);
  Regs.ES := SEG(LongNameArray[1]);
  Regs.DI := OFS(LongNameArray[1]);
  MSDOS(Regs);
  IF Regs.FLAGS AND FCARRY <> 0 THEN
  EXIT
  ELSE
  BEGIN
    LongPath := StrPas(Addr(LongNameArray));
    LFN_FileSplit(LongPath, LongDir, LongName, Ext);
    LongName := LongName + Ext;
    FSPLIT(SName, Dir, Name, Ext);
    LFN_GetLongFileName := (LENGTH(LongName) > 0) AND (Name + Ext <> LongName);
  END;
END; {LFN_GetLongFileName}


FUNCTION LFN_GetShortFileName(LongName : STRING; VAR ShortName : String12) : BOOLEAN;
VAR Regs       : REGISTERS;
    LongPath   : STRING;
    SNameArray : ARRAY[1..261] OF CHAR;
    ShortPath  : STRING;
    SDir       : PathStr;
    SName      : NameStr;
    SExt       : ExtStr;

BEGIN
  LFN_GetShortFileName := FALSE;
  ShortName := '';

  LongPath := LongName + #0;
  Regs.AX := $7160;
  Regs.CL := $1;
  Regs.CH := $0;
  Regs.DS := SEG(LongPath[1]);
  Regs.SI := OFS(LongPath[1]);
  Regs.ES := SEG(SNameArray[1]);
  Regs.DI := OFS(SNameArray[1]);
  MSDOS(Regs);
  IF Regs.FLAGS AND FCARRY <> 0 THEN
  EXIT
  ELSE
  BEGIN
    ShortPath := StrPas(Addr(SNameArray));
    FSplit(ShortPath, SDir, SName, SExt);
    ShortName := SName + SExt;
    LFN_GetShortFileName := TRUE;
  END;
END; {LFN_GetShortFileName}


PROCEDURE LFN_FileSplit(Path : STRING; VAR Dir, Name : STRING; VAR Ext : EXTSTR);

VAR LowerB : BYTE;
    i      : INTEGER;
    
BEGIN
  Dir  := '';
  Name := '';
  Ext  := '';
  LowerB := Maximum(1, LENGTH(Path) - 4);
  
  FOR i := LENGTH(Path) DOWNTO  LowerB DO
  BEGIN
    IF Path[i] = '.' THEN
    BEGIN
      Ext  := COPY(Path, i, 4);
      Path := COPY(Path, 1, Maximum(1, PRED(i)));
      Break;
    END;
  END;
  FOR i := LENGTH(Path) DOWNTO  1 DO
  BEGIN
    IF Path[i] = '\' THEN
    BEGIN
      Name := COPY(Path, SUCC(i), LENGTH(Path) - PRED(i));
      Dir  := COPY(Path, 1, Maximum(1, i));
      Break;
    END;
  END;
  IF Name = '' Then
  Name := Path;
END; {LFN_FileSplit}



BEGIN {WIN95LFN Initialization}
  WinVersion := GetWinVersion;
  LFNSupport := LO(WinVersion) >= 4;
END.  {WIN95LFN Initialization}
